--[[
Copyright (C) GtX (Andy), 2025

Author: GtX | Andy
Date: 27.04.2025
Revision: FS25-01

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]

ConstructionScreenExtension = {}

local ConstructionScreenExtension_mt = Class(ConstructionScreenExtension)

local modName = g_currentModName
local modDirectory = g_currentModDirectory

local versionString = "0.0.0.0"
local buildId = 1

local emptyTable = {}
local validationFail

local consoleCommandsGtX = StartParams.getIsSet("consoleCommandsGtX") -- Extra GtX console commands

function ConstructionScreenExtension.new(customMt)
    local self = setmetatable({}, customMt or ConstructionScreenExtension_mt)

    self.modName = modName
    self.modDirectory = modDirectory

    self.versionString = versionString
    self.buildId = buildId

    self.angleText = g_i18n:getText("cse_angle", modName)
    self.quickCopyText = g_i18n:getText("cse_quickCopy", modName)
    self.customiseText = g_i18n:getText("input_CONSTRUCTION_SHOW_CONFIGS")

    self.initalised = false
    self.debugEnabled = buildId == 0

    return self
end

function ConstructionScreenExtension:load(typeManager)
    if self.initalised then
        return false
    end

    ConstructionBrushCustomFieldDialog.register()

    if typeManager:getClassObjectByTypeName("customField") ~= nil then
        if self.debugEnabled then
            Logging.devInfo("[ConstructionScreenExtension]  Adding feature - Custom Field")
        end
    end

    if ConstructionScreenExtension.extendConstructionBrushSelect(self, typeManager, g_constructionScreen) then
        if self.debugEnabled then
            Logging.devInfo("[ConstructionScreenExtension]  Adding feature - Quick Copy")
            Logging.devInfo("[ConstructionScreenExtension]  Adding feature - Configure Existing")
        end
    end

    if ConstructionScreenExtension.extendConstructionBrushPlaceable(self, typeManager, g_constructionScreen) then
        if self.debugEnabled then
            Logging.devInfo("[ConstructionScreenExtension]  Adding feature - Price Fix")
            Logging.devInfo("[ConstructionScreenExtension]  Adding feature - Angle Display")
        end
    end

    -- Testing and content creator commands. Use -cheats or -consoleCommandsGtX start parameter
    if consoleCommandsGtX or g_addCheatCommands then
        addConsoleCommand("gtxCustomFieldNoCostPlacement", "Toggles the no cost option for Custom Field placement", "consoleCommandToggleNoCostPlacement", self)
    end

    self.initalised = true

    return true
end

function ConstructionScreenExtension:delete()
    if consoleCommandsGtX or g_addCheatCommands then
        removeConsoleCommand("gtxCustomFieldNoCostPlacement")
    end

    self.initalised = false
end

function ConstructionScreenExtension:consoleCommandToggleNoCostPlacement()
    if PlaceableCustomField == nil or g_currentMission == nil or not self.initalised then
        return "Not available!"
    end

    if not g_currentMission.isMasterUser then
        return "Not possible, console command is an Admin only feature!"
    end

    PlaceableCustomField.NO_COST_PLACEMENT = not PlaceableCustomField.NO_COST_PLACEMENT

    return "No cost placement active = " .. tostring(PlaceableCustomField.NO_COST_PLACEMENT)
end

function ConstructionScreenExtension:onPlaceableChanged(success)
    if self.placeableChangedCallback ~= nil then
        self.placeableChangedCallback(success)
        self.placeableChangedCallback = nil
    end

    if self.debugEnabled then
        Logging.devInfo("[ConstructionScreenExtension]  onPlaceableChanged - " .. tostring(success))
    end
end

function ConstructionScreenExtension.extendConstructionBrushSelect(extension, typeManager, constructionScreen)
    local classObject = typeManager:getClassObjectByTypeName("select")

    if classObject ~= nil then
        local oldNew = classObject.new

        classObject.new = function(subclass_mt, cursor)
            local brush = oldNew(subclass_mt, cursor)

            if extension ~= nil and extension.initalised then
                -- Quick Select
                brush.supportsSecondaryButton = true
                brush.supportsSecondaryDragging = false

                -- Configure Existing
                brush.supportsFourthButton = true
            end

            return brush
        end

        -- Quick Copy
        classObject.onButtonSecondary = function(brush)
            if extension ~= nil and extension.initalised then
                local placeable = brush.lastPlaceable

                if placeable ~= nil and placeable.storeItem ~= nil then
                    local storeItem = placeable.storeItem

                    if storeItem.brush ~= nil then
                        local selectedItem, categoryIndex, tabIndex, itemIndex = ConstructionScreenExtension.getSelectedConstructionScreenItem(constructionScreen, storeItem)

                        if selectedItem ~= nil then
                            -- Scroll to the items UI element
                            ConstructionScreenExtension.scrollToConstructionScreenItem(constructionScreen, categoryIndex, tabIndex, itemIndex)

                            -- Create new brush instance
                            local brush = selectedItem.brushClass.new(nil, constructionScreen.cursor)

                            -- Copy the placeable rotation
                            local dx, _, dz = localDirectionToWorld(placeable.rootNode, 0, 0, 1)
                            local rotY = MathUtil.getYRotationFromDirection(dx, dz)

                            if selectedItem.brushParameters ~= nil then
                                local configurations = placeable.configurations
                                local configurationData = placeable.configurationData

                                brush:setStoreItem(storeItem, configurations, configurationData)
                                brush:setParameters(unpack(selectedItem.brushParameters))

                                if brush.placeableHasConfigs then
                                    local oldLoadPlaceable = brush.loadPlaceable
                                    local posX, posY, posZ = placeable:getPosition()

                                    -- Apply configurations when loading
                                    brush.loadPlaceable = function(_, data)
                                        brush.loadPlaceable = oldLoadPlaceable

                                        if data == nil and posX ~= nil then
                                            data = PlaceableLoadingData.new()

                                            data:setConfigurations(configurations)
                                            data:setConfigurationData(configurationData)
                                            data:setPosition(posX, posY, posZ)
                                        end

                                        oldLoadPlaceable(brush, data)
                                    end
                                end

                                brush.uniqueIndex = selectedItem.uniqueIndex
                            end

                            constructionScreen.destructMode = false

                            constructionScreen.cursor:setRotation(rotY)
                            constructionScreen.cursor.lastActionFrame = g_time + 250 -- Avoid getting stuck rotating when pressing clone key on mouse

                            constructionScreen:setBrush(brush, true)
                        else
                            g_currentMission:showBlinkingWarning(g_i18n:getText("warning_actionNotAllowedNow"))
                        end
                    end
                end
            end
        end

        classObject.getButtonSecondaryText = function(brush)
            if extension ~= nil and extension.initalised then
                local placeable = brush.lastPlaceable

                if placeable ~= nil and placeable.storeItem ~= nil and placeable.storeItem.canBeSold and placeable.spec_trainSystem == nil and placeable.spec_riceField == nil then
                    local storeItemBrush = placeable.storeItem.brush

                    if storeItemBrush ~= nil and (storeItemBrush.type == "placeable" or storeItemBrush.type == "husbandry") then
                        return extension.quickCopyText
                    end
                end
            end

            return nil
        end

        -- Configure Existing
        classObject.onButtonFourth = function(brush)
            local existingPlaceable = brush.lastPlaceable
            local valid = false

            if ConstructionScreenExtension.getCanPlaceableBeConfigured(existingPlaceable) then
                local selectedItem, categoryIndex, tabIndex, itemIndex = ConstructionScreenExtension.getSelectedConstructionScreenItem(constructionScreen, existingPlaceable.storeItem)

                if selectedItem ~= nil then
                    -- Scroll to the items UI element
                    ConstructionScreenExtension.scrollToConstructionScreenItem(constructionScreen, categoryIndex, tabIndex, itemIndex)

                    -- Create new brush instance
                    local brush = selectedItem.brushClass.new(nil, constructionScreen.cursor)

                    -- Copy the existing placeable rotation
                    local dx, _, dz = localDirectionToWorld(existingPlaceable.rootNode, 0, 0, 1)
                    local rotY = MathUtil.getYRotationFromDirection(dx, dz)

                    if selectedItem.brushParameters ~= nil then
                        local configurations = existingPlaceable.configurations
                        local configurationData = existingPlaceable.configurationData

                        brush:setStoreItem(existingPlaceable.storeItem, configurations, configurationData)
                        brush:setParameters(unpack(selectedItem.brushParameters))

                        if brush.placeableHasConfigs then
                            local oldLoadPlaceable = brush.loadPlaceable
                            local posX, posY, posZ = existingPlaceable:getPosition()

                            -- Apply configurations when loading
                            brush.loadPlaceable = function(_, data)
                                brush.loadPlaceable = oldLoadPlaceable

                                if data == nil and posX ~= nil then
                                    existingPlaceable:removeFromPhysics()
                                    existingPlaceable:setVisibility(false)

                                    data = PlaceableLoadingData.new()

                                    data:setConfigurations(configurations)
                                    data:setConfigurationData(configurationData)
                                    data:setPosition(posX, posY, posZ)
                                end

                                oldLoadPlaceable(brush, data)
                            end

                            local oldLoadedPlaceable = brush.loadedPlaceable

                            brush.loadedPlaceable = function(_, placeable, loadingState, args)
                                oldLoadedPlaceable(brush, placeable, loadingState, args)

                                if brush.placeable ~= nil then
                                    if not constructionScreen.configsBox:getIsVisible() then
                                        constructionScreen:onShowConfigs()
                                    end

                                    brush.existingPlaceable = existingPlaceable
                                end
                            end
                        else
                            brush:delete()

                            return
                        end

                        brush.uniqueIndex = selectedItem.uniqueIndex
                    end

                    constructionScreen.destructMode = false

                    constructionScreen.cursor:setRotation(rotY)
                    constructionScreen:setBrush(brush, true)

                    valid = true
                end

                if not valid then
                    g_currentMission:showBlinkingWarning(g_i18n:getText("warning_actionNotAllowedNow"))
                end
            end
        end

        classObject.getButtonFourthText = function(brush)
            if extension ~= nil and extension.initalised and brush.lastPlaceable then
                if ConstructionScreenExtension.getCanPlaceableBeConfigured(brush.lastPlaceable) then
                    return extension.customiseText
                end
            end

            return nil
        end

        return true
    end

    return false
end

function ConstructionScreenExtension.extendConstructionBrushPlaceable(extension, typeManager)
    local classObject = typeManager:getClassObjectByTypeName("placeable")

    if classObject ~= nil then
        local oldUpdatePlaceablePosition = classObject.updatePlaceablePosition

        -- Display the placeables angle / rotation to assist with placing
        classObject.updatePlaceablePosition = function(brush, ...)
            oldUpdatePlaceablePosition(brush, ...)

            if extension and extension.initalised then
                if brush.errorText == nil and brush.placeable ~= nil and getVisibility(brush.placeable.rootNode) then
                    local cursorRotY = brush.cursor:getRotation()
                    local rotY = brush:getSnappedRotation(cursorRotY)

                    g_currentMission:addExtraPrintText(string.format(extension.angleText, math.deg(math.abs(rotY - math.pi))))
                end
            end
        end

        -- Base game does not pass the 'configurations' parameter so fixed here to avoid nasty surprises after placement
        classObject.getPrice = function (brush)
            return g_currentMission.economyManager:getBuyPrice(brush.storeItem, brush.configurations, nil) + brush:getDisplacementCost()
        end

        return true
    end

    return false
end

function ConstructionScreenExtension.scrollToConstructionScreenItem(constructionScreen, categoryIndex, tabIndex, itemIndex)
    if constructionScreen ~= nil and itemIndex ~= nil then
        constructionScreen.itemList:setSelectedIndex(1, false, true) -- Select the first item to try and make slider position reliable when updating to new tab and item

        constructionScreen:setCurrentCategory(categoryIndex, nil)
        constructionScreen.currentTab = tabIndex or 1
        constructionScreen.subCategorySelector:setState(constructionScreen.currentTab, true)

        local numItems = #constructionScreen.items[constructionScreen.currentCategory][constructionScreen.currentTab]

        if numItems > 0 then
            if numItems >= itemIndex then
                constructionScreen.itemList:setSelectedIndex(itemIndex, true, false)
            else
                constructionScreen.itemList:setSelectedIndex(1, true, false) -- Fallback
            end
        else
            constructionScreen:assignItemAttributeData(nil) -- If nothing then clear any existing info
        end

        constructionScreen:setBrush(constructionScreen.selectorBrush, true)
        constructionScreen:updateMenuState()
    end
end

function ConstructionScreenExtension.getSelectedConstructionScreenItem(constructionScreen, storeItem)
    if constructionScreen ~= nil and storeItem ~= nil then
        local categoryIndex = storeItem.brush.category.index
        local tabIndex = storeItem.brush.tab.index

        local items = constructionScreen.items[categoryIndex][tabIndex]

        if items ~= nil then
            for itemIndex, item in ipairs (items) do
                if item.storeItem == storeItem then
                    return item, categoryIndex, tabIndex, itemIndex
                end
            end
        end
    end

    return nil
end

function ConstructionScreenExtension.getCanPlaceableBeConfigured(placeable)
    if placeable == nil or placeable.storeItem == nil or placeable.storeItem.configurations == nil or not placeable.storeItem.canBeSold then
        return false
    end

    if placeable.isDeleted or placeable.isPreplaced or placeable.spec_trainSystem ~= nil or placeable.spec_riceField ~= nil then
        return false
    end

    for _, configs in pairs(placeable.storeItem.configurations) do
        if #configs > 1 then
            return true
        end
    end

    return false
end

local function validateMod()
    local mod = g_modManager:getModByName(modName)

    if mod == nil or g_iconGenerator ~= nil or g_isEditor then
        return false
    end

    versionString = mod.version or versionString

    if mod.modName == "FS25_ConstructionScreenExtension" or mod.modName == "FS25_ConstructionScreenExtension_update" then
        if mod.author ~= nil and #mod.author == 3 then
            return true
        end
    end

    local validationText = "This mod is outdated and not working properly. Please update the mod first!"

    if g_i18n:hasText("ui_modsCannotActivateBlockedModUpdate") then
        validationText = g_i18n:getText("ui_modsCannotActivateBlockedModUpdate")
    end

    validationText = string.format("- %s -\n\n%s", mod.modName, validationText)

    validationFail = {
        startUpdateTime = 2000,

        update = function(self, dt)
            self.startUpdateTime = self.startUpdateTime - dt

            if self.startUpdateTime < 0 then
                if g_dedicatedServer == nil then
                    if not g_gui:getIsGuiVisible() then
                        local yesText = g_i18n:getText("button_modHubDownload")
                        local noText = g_i18n:getText("button_ok")

                        YesNoDialog.show(self.openModHubLink, nil, validationText, mod.title, yesText, noText, DialogElement.TYPE_WARNING)
                    end
                else
                    print("\n" .. validationText .. "\n    - https://farming-simulator.com/mods.php?&title=fs2025&filter=org&org_id=129652&page=0" .. "\n")
                    self.openModHubLink(false)
                end
            end
        end,

        openModHubLink = function(yes)
            if yes then
                openWebFile("mods.php?title=fs2025&filter=org&org_id=129652&page=0", "")
            end

            removeModEventListener(validationFail)
            validationFail = nil
        end
    }

    addModEventListener(validationFail)

    return false
end

local function constructionBrushTypeManager_unloadMapData(self)
    if g_constructionScreenExtension ~= nil then
        g_constructionScreenExtension:delete()
    end

    if g_globalMods ~= nil then
        g_globalMods.constructionScreenExtension = nil
    end

    g_constructionScreenExtension = nil

    -- Cleanup pointer and spec from this environment
    PlaceableRiceFieldSpec = nil
    PlaceableRiceField = nil
end

local function constructionScreen_onShowConfigs(self, superFunc)
    if self.configsBox:getIsVisible() then
        local brush = self.brush

        if brush ~= nil and brush.existingPlaceable ~= nil and brush.configurations ~= nil then
            local placeable = brush.existingPlaceable

            local function onConfirmChangesCallback(applyChanges)
                brush.existingPlaceable = nil

                if applyChanges then
                    if g_currentMission:getHasPlayerPermission("buyPlaceable") then
                        local posX, posY, posZ = placeable:getPosition()

                        local dx, _, dz = localDirectionToWorld(placeable.rootNode, 0, 0, 1)
                        local rotY = MathUtil.getYRotationFromDirection(dx, dz)

                        local placeableBuyData = BuyPlaceableData.new()

                        placeableBuyData:setStoreItem(brush.storeItem)
                        placeableBuyData:setConfigurations(brush.configurations or {})
                        placeableBuyData:setConfigurationData(brush.configurationData)
                        placeableBuyData:setPosition(posX, posY, posZ)
                        placeableBuyData:setRotation(0, rotY, 0)
                        placeableBuyData:setIsFreeOfCharge(false)
                        placeableBuyData:setOwnerFarmId(g_localPlayer.farmId)
                        placeableBuyData:setDisplacementCosts(0)
                        placeableBuyData:setModifyTerrain(false)
                        placeableBuyData:updatePrice()

                        if g_constructionScreenExtension ~= nil then
                            -- Handle hiding of 'MessageDialog' and error messages if failed server side
                            g_constructionScreenExtension.placeableChangedCallback = function(success)
                                MessageDialog.hide()

                                if success == false then
                                    InfoDialog.show(g_i18n:getText("shop_messageConfigurationChangeFailed"))
                                end
                            end
                        end

                        g_client:getServerConnection():sendEvent(ChangePlaceableConfigEvent.new(placeable, placeableBuyData))
                    else
                        g_currentMission:showBlinkingWarning(g_i18n:getText("shop_messageConfigurationChangeFailed") .. "\n" .. g_i18n:getText("shop_messageNoPermissionGeneral"))
                        applyChanges = false
                    end
                end

                -- Update GUI to correctly hide configsBox
                superFunc(self)

                -- Show original placeable again in case the update fails server side
                if placeable ~= nil and not placeable.isDeleted then
                    placeable:addToPhysics()
                    placeable:setVisibility(true)
                end

                -- Return to select brush and cleanup temp placeable used to create buy data
                if not self.brush.isSelector then
                    self:setBrush(self.selectorBrush)
                end

                if applyChanges then
                    -- Only allow 'MessageDialog' to show for a maximum of 8 seconds, anymore and there is serious network issues anyway
                    local function updateCallback(dt, endTimeSec)
                        if endTimeSec ~= nil and endTimeSec <= getTimeSec() then
                            MessageDialog.hide()
                        end
                    end

                    MessageDialog.show(g_i18n:getText("setting_applyingChanges"), updateCallback, nil, DialogElement.TYPE_LOADING, false, getTimeSec() + 8)
                end
            end

            local configurationsChanged = false

            if brush.configurations ~= nil then
                for configName, configValue in pairs(brush.configurations) do
                    if placeable.configurations[configName] ~= configValue then
                        configurationsChanged = true

                        break
                    end
                end
            end

            if not configurationsChanged then
                configurationsChanged = ConfigurationUtil.getConfigurationDataHasChanged(placeable.configFileName, brush.configurationData, placeable.configurationData)
            end

            if configurationsChanged then
                local existingPrice = placeable.price or g_currentMission.economyManager:getBuyPrice(placeable.storeItem, placeable.configurations, nil) or 0
                local price = g_currentMission.economyManager:getBuyPrice(placeable.storeItem, brush.configurations, nil) or 0

                local upgradePrice = 0

                if existingPrice < price then
                    upgradePrice = price - existingPrice
                end

                if upgradePrice == 0 or g_currentMission:getMoney() - upgradePrice >= 0 then
                    local text = g_i18n:getText("shop_messageConfigurationChanged") .. "\n" .. g_i18n:getText("ui_total") .. ": %s"
                    local yesText = g_i18n:getText("button_apply")
                    local noText = g_i18n:getText("button_cancel")

                    YesNoDialog.show(onConfirmChangesCallback, nil, text:format(g_i18n:formatMoney(upgradePrice, 0, true, true)), placeable:getName(), yesText, noText, DialogElement.TYPE_INFO)
                else
                    InfoDialog.show(g_i18n:getText("shop_messageConfigurationChangeFailed") .. "\n" .. g_i18n:getText("shop_messageNotEnoughMoneyToBuy"), onConfirmChangesCallback, nil, nil, nil, nil, false)
                end
            else
                onConfirmChangesCallback(false) -- No changes, cleanup and ignore.
            end

            return
        end
    end

    superFunc(self)
end

local function constructionScreen_onClose(self, element)
    if g_currentMission == nil or g_currentMission.placeableSystem == nil then
        return
    end

    local placeables = g_currentMission.placeableSystem.placeables or emptyTable

    for _, placeable in ipairs(placeables) do
        placeable.overlayColorNodes = nil -- Base game bug fix with some placeable objects
    end
end

local function animalClusterUpdateEvent_readStream(self, superFunc, streamId, connection)
    local valid = streamReadBool(streamId)

    if valid then
        superFunc(self, streamId, connection)
    end
end

local function animalClusterUpdateEvent_writeStream(self, superFunc, streamId, connection)
    local valid = self.owner ~= nil and (self.owner.getIsSynchronized == nil or self.owner:getIsSynchronized())

    if streamWriteBool(streamId, valid) then
        superFunc(self, streamId, connection)
    end
end

local function init()
    if g_globalMods == nil then
        g_globalMods = {}
    end

    if g_globalMods.constructionScreenExtension ~= nil then
        Logging.error("Validation of '%s' failed, script set has already been loaded by '%s'.", modName, g_globalMods.constructionScreenExtension.modName or "Unknown")

        return false
    end

    if validateMod() then
        -- There is no way to get the original functions if another mod overwrites first. Would be a great option @Giants to retain the 'original function' for mods like this :-)
        -- Because source loads scripts into a mods environment as long as the path is absolute it means I can use the unchanged functions and break nothing.
        PlaceableRiceFieldSpec = PlaceableRiceField
        local appPath = getAppBasePath ~= nil and getAppBasePath() or ""
        source(appPath .. "dataS/scripts/placeables/specializations/PlaceableRiceField.lua")

        source(modDirectory .. "scripts/events/ChangePlaceableConfigEvent.lua")
        source(modDirectory .. "scripts/events/PlaceableSetIsReconfiguratingEvent.lua")
        source(modDirectory .. "scripts/events/PlaceableCustomFieldEvent.lua")
        source(modDirectory .. "scripts/events/PlaceableCustomFieldAnswerEvent.lua")

        source(modDirectory .. "scripts/gui/ConstructionBrushCustomFieldDialog.lua")

        local constructionBrushTypeManager = g_constructionBrushTypeManager

        constructionBrushTypeManager:addBrushType("customField", "ConstructionBrushCustomField", modDirectory .. "scripts/construction/ConstructionBrushCustomField.lua", modName)

        local constructionScreenExtension = ConstructionScreenExtension.new()

        if constructionScreenExtension:load(constructionBrushTypeManager) then
            g_constructionScreenExtension = constructionScreenExtension

            g_globalMods.constructionScreenExtension = {
                modName = modName,
                versionString = versionString,
                buildId = buildId
            }

            -- Cleanup on exit
            ConstructionBrushTypeManager.unloadMapData = Utils.appendedFunction(ConstructionBrushTypeManager.unloadMapData, constructionBrushTypeManager_unloadMapData)

            -- Customise Mode
            ConstructionScreen.onShowConfigs = Utils.overwrittenFunction(ConstructionScreen.onShowConfigs, constructionScreen_onShowConfigs)

            -- Not listed as a feature but fixes a base game bug / error where the cached nodes throw errors when something such as a greenhouse is turned on/off.
            -- This is because temporary visibility nodes such as mushrooms for example are deleted.
            -- Example: Open construction screen > highlight greenhouse that is active > leave screen and deactivate greenhouse > open screen and highlight  =  error from missing nodes
            ConstructionScreen.onClose = Utils.appendedFunction(ConstructionScreen.onClose, constructionScreen_onClose)

            -- Fix for the Animal Clusters trying to broadcast this event before the placeable is synchronised
            AnimalClusterUpdateEvent.readStream = Utils.overwrittenFunction(AnimalClusterUpdateEvent.readStream, animalClusterUpdateEvent_readStream)
            AnimalClusterUpdateEvent.writeStream = Utils.overwrittenFunction(AnimalClusterUpdateEvent.writeStream, animalClusterUpdateEvent_writeStream)

            -- There is no mod check for brush class names when loading placeables, this warning is for Giants test team in case mod is loaded with update suffix.
            if string.endsWith(modName, "_update") then
                Logging.warning("[%s] One or more placeables included with this mod require the custom environment 'FS25_ConstructionScreenExtension' and will not load correctly when using the '_update' suffix.", modName)
            end
        end
    else
        -- Failed to load so stop placeable loading because there is no localisation checks for the placeable brush
        StoreManager.addModStoreItem = Utils.overwrittenFunction(StoreManager.addModStoreItem, function(storeManager, superFunc, xmlFilename, baseDir, customEnvironment, ...)
            if customEnvironment == modName then
                return
            end

            superFunc(storeManager, xmlFilename, baseDir, customEnvironment, ...)
        end)

        Logging.error("[%s] Failed to initialise / validate mod, make sure you are using the latest release available on the Giants Official Mod Hub with no file or zip name changes!", modName)
    end
end

init()
