--[[
	RoundBalerExtension.lua
	
	Author: 	Ifko[nator]
	Date: 		08.03.2025
	Version: 	5.1
	
	Changelog:	v1.0 @24.12.2018 - initial implementation in FS 19
				---------------------------------------------------------------------------------------------------------------------------
				v1.5 @19.04.2020 - fix for straw harvest addon and the cruise control will not longer set to max, after bale unloading
				---------------------------------------------------------------------------------------------------------------------------
				v2.0 @12.08.2020 - added support for the fastBale from the Kverneland DLC. The Tractor will no longer stop with tihs baler.
				---------------------------------------------------------------------------------------------------------------------------
				v3.0 @28.09.2020 - added possibility to change the unload mode to manual. For fields with an big slope.
				---------------------------------------------------------------------------------------------------------------------------
				v4.0 @23.02.2021 - convert to FS 22
				---------------------------------------------------------------------------------------------------------------------------
				v4.1 @26.02.2022 - fix for square balers. They are no longer limited to 8 km/h if they had reached once 90% fill level
				---------------------------------------------------------------------------------------------------------------------------
				v4.2 @24.03.2022 - brakelights work now again, when baler is turned on
				---------------------------------------------------------------------------------------------------------------------------
				v4.3 @21.08.2022 - added support for vermeer DLC
				---------------------------------------------------------------------------------------------------------------------------
				v4.4 @24.03.2022 - added support for Hof Bergmann Version 1.2.0.0
				---------------------------------------------------------------------------------------------------------------------------
				v5.0 @24.11.2024 - convert to FS 25
				---------------------------------------------------------------------------------------------------------------------------
				v5.1 @08.03.2025 - adaption for different bale sizes for the collect mode
]]

RoundBalerExtension = {};

function RoundBalerExtension.initSpecialization()
	Vehicle.xmlSchemaSavegame:register(XMLValueType.INT, "vehicles.vehicle(?).roundBalerExtension#wrappedBaleCount", "Current wrapped bale count.", 0);
end;


function RoundBalerExtension.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(Drivable, specializations);
end;

function RoundBalerExtension.registerEventListeners(vehicleType)
	local functionNames = {
		"onLoad",
		"onUpdate",
		"onReadStream",
		"onWriteStream",
		"saveToXMLFile"
	};
	
	for _, functionName in ipairs(functionNames) do
		SpecializationUtil.registerEventListener(vehicleType, functionName, RoundBalerExtension);
	end;
end;

function RoundBalerExtension.registerFunctions(vehicleType)
	local newFunctions = {
		"lockVehicle",
		"resetForNextBale",
		"setBaleToUnload",
		"setCruiseControlWasOn",
		"handleVehicleTransmission",
		"foundRoundBaler",
		"getIsAllowedToDriveOffAfterUnloading"
	};
	
	for _, newFunction in ipairs(newFunctions) do
		SpecializationUtil.registerFunction(vehicleType, newFunction, RoundBalerExtension[newFunction]);
	end;
end;

function RoundBalerExtension:onLoad(savegame)
	local specRoundBalerExtension = self.spec_roundBalerExtension;
	
	specRoundBalerExtension.hasBaleToUnload = false;
	specRoundBalerExtension.cruiseControlWasOn = false;
	
	specRoundBalerExtension.wrappedBaleCount = 0;
	specRoundBalerExtension.driveOffTimer = 0;

	if savegame ~= nil and not savegame.resetVehicles then
		specRoundBalerExtension.wrappedBaleCount = savegame.xmlFile:getValue(savegame.key .. ".roundBalerExtension#wrappedBaleCount", specRoundBalerExtension.wrappedBaleCount);
	end;

	specRoundBalerExtension.driveOffTimerMax125 = 1050;
	specRoundBalerExtension.driveOffTimerMax150 = 1150;
	specRoundBalerExtension.driveOffTimerMax180 = 1300;

	specRoundBalerExtension.driveOffTimerMax = specRoundBalerExtension.driveOffTimerMax125;
end;

function RoundBalerExtension:onUpdate(deltaTime, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	if self:getIsActive() then
		local specAttacherJoints = self.spec_attacherJoints;
		local specRoundBalerExtension = self.spec_roundBalerExtension;
		local specBaler = self.spec_baler;

		if specBaler ~= nil then
			local specDrivable = self.spec_drivable;
			local specWheels = self.spec_wheels;

			if self.speedLimitBackup == nil then
				self.speedLimitBackup = self.speedLimit;
			end;

			self:handleVehicleTransmission(specBaler, self, nil, specRoundBalerExtension, specDrivable, specWheels, nil, deltaTime);
		end;
		
		if specAttacherJoints ~= nil then
			for _, implement in pairs(specAttacherJoints.attachedImplements) do
				if implement ~= nil and implement.object ~= nil then
					local roundBaler = implement.object;
					local controlledTractor = roundBaler:getAttacherVehicle();

					local specDrivable = controlledTractor.spec_drivable;
					local specWheelsTractor = controlledTractor.spec_wheels;

					local specBaler = roundBaler.spec_baler;
					local specWheelsRoundBaler = roundBaler.spec_wheels;

					if roundBaler.speedLimitBackup == nil then
						roundBaler.speedLimitBackup = roundBaler.speedLimit;
					end;

					self:handleVehicleTransmission(specBaler, controlledTractor, roundBaler, specRoundBalerExtension, specDrivable, specWheelsTractor, specWheelsRoundBaler, deltaTime);
				end;
			end;
		end;
	end;
end;

function RoundBalerExtension:saveToXMLFile(xmlFile, key)
	local specRoundBalerExtension = self.spec_roundBalerExtension;

	if specRoundBalerExtension.wrappedBaleCount > 0 then
		xmlFile:setValue(key .. "#wrappedBaleCount", specRoundBalerExtension.wrappedBaleCount);
	end;
end;

function RoundBalerExtension:handleVehicleTransmission(specBaler, controlledTractor, roundBaler, specRoundBalerExtension, specDrivable, specWheelsTractor, specWheelsRoundBaler, deltaTime)
	roundBaler = Utils.getNoNil(roundBaler, controlledTractor);

	local specToggleableBaleWrapper = roundBaler.spec_toggleableBaleWrapper;

	local isAllowedToDriveOffAfterUnloading = self:getIsAllowedToDriveOffAfterUnloading(specRoundBalerExtension, specToggleableBaleWrapper);

	if self:foundRoundBaler(specBaler, roundBaler) then
		--## we have an round baler attached :)
		
		if not isAllowedToDriveOffAfterUnloading then
			local maxTimer = specRoundBalerExtension["driveOffTimerMax" .. specBaler.baleTypes[specBaler.currentBaleTypeIndex].diameter * 100];
			
			if specRoundBalerExtension.driveOffTimer < maxTimer then
				specRoundBalerExtension.driveOffTimer = math.clamp((specRoundBalerExtension.driveOffTimer + deltaTime), 0, maxTimer);
			else
				self:lockVehicle(specRoundBalerExtension, specDrivable, specWheelsTractor, specWheelsRoundBaler, controlledTractor);
			end;
		elseif roundBaler:getFillUnitFillLevelPercentage(specBaler.fillUnitIndex) >= 0.95 and roundBaler:getFillUnitFreeCapacity(specBaler.fillUnitIndex) > 0 then
			roundBaler.speedLimit = 8;
		elseif roundBaler:getFillUnitFreeCapacity(specBaler.fillUnitIndex) == 0 then
			self:setBaleToUnload(specRoundBalerExtension, roundBaler);
			self:lockVehicle(specRoundBalerExtension, specDrivable, specWheelsTractor, specWheelsRoundBaler, controlledTractor);
		elseif specBaler.unloadingState ~= Baler.UNLOADING_CLOSED then
			self:lockVehicle(specRoundBalerExtension, specDrivable, specWheelsTractor, specWheelsRoundBaler, controlledTractor);
		elseif specBaler.unloadingState == Baler.UNLOADING_CLOSED then
			if specRoundBalerExtension.hasBaleToUnload then
				if roundBaler.spec_baleWrapper ~= nil and specToggleableBaleWrapper ~= nil and specToggleableBaleWrapper.currentDropModeState == ToggleableBaleWrapper.DROP_MODE_COLLECT then
					specRoundBalerExtension.wrappedBaleCount = specRoundBalerExtension.wrappedBaleCount + 1;
				end;
				
				if #specBaler.bales == 0 then
					self:resetForNextBale(specRoundBalerExtension, controlledTractor, roundBaler);
				else
					self:lockVehicle(specRoundBalerExtension, specDrivable, specWheelsTractor, specWheelsRoundBaler, controlledTractor);
				end;
			end;
		end;

		if roundBaler.spec_baleWrapper ~= nil and roundBaler.spec_baleWrapper.baleWrapperState == BaleWrapper.STATE_WRAPPER_RESETTING_PLATFORM and specRoundBalerExtension.wrappedBaleCount ==  2 then
			specRoundBalerExtension.wrappedBaleCount = 0;
			specRoundBalerExtension.driveOffTimer = 0;

			self:resetForNextBale(specRoundBalerExtension, controlledTractor, roundBaler);
		end;
	end;
end;

function RoundBalerExtension:getIsAllowedToDriveOffAfterUnloading(specRoundBalerExtension, specToggleableBaleWrapper)
	local isAllowedToDriveOffAfterUnloading = true;
	
	if specToggleableBaleWrapper ~= nil then
		if specToggleableBaleWrapper.currentDropModeState == ToggleableBaleWrapper.DROP_MODE_COLLECT then
			isAllowedToDriveOffAfterUnloading = specRoundBalerExtension.wrappedBaleCount <= 1;
		else
			if specRoundBalerExtension.wrappedBaleCount > 0 then
				specRoundBalerExtension.wrappedBaleCount = 0;
			end;
		end;
	end;

	return isAllowedToDriveOffAfterUnloading;
end;

function RoundBalerExtension:foundRoundBaler(specBaler, roundBaler)
	return specBaler ~= nil and specBaler.hasUnloadingAnimation and roundBaler.getIsTurnedOn ~= nil and roundBaler:getIsTurnedOn() and not specBaler.nonStopBaling;
end;

function RoundBalerExtension:resetForNextBale(specRoundBalerExtension, controlledTractor, roundBaler)
	specRoundBalerExtension.hasBaleToUnload = false;

	if specRoundBalerExtension.cruiseControlWasOn then	
		controlledTractor:setCruiseControlState(Drivable.CRUISECONTROL_STATE_ACTIVE);

		self:setCruiseControlWasOn(false, false);
	end;
end;

function RoundBalerExtension:lockVehicle(specRoundBalerExtension, specDrivable, specWheelsTractor, specWheelsRoundBaler, controlledTractor)
	if self.isServer then
		for _, wheel in pairs(specWheelsTractor.wheels) do
			setWheelShapeProps(wheel.node, wheel.physics.wheelShape, 0, math.huge, 0, 0);
		end;
		
		if specWheelsRoundBaler ~= nil then
			for _, wheel in pairs(specWheelsRoundBaler.wheels) do
				setWheelShapeProps(wheel.node, wheel.physics.wheelShape, 0, math.huge, 0, 0);
			end;
		end;
	end;

	controlledTractor:setBrakeLightsVisibility(true);

	if specDrivable.cruiseControl.state ~= Drivable.CRUISECONTROL_STATE_OFF then
		self:setCruiseControlWasOn(true, false);
		
		controlledTractor:setCruiseControlState(Drivable.CRUISECONTROL_STATE_OFF);
	end;
end;

function RoundBalerExtension:setBaleToUnload(specRoundBalerExtension, roundBaler)
	if not specRoundBalerExtension.hasBaleToUnload then
		specRoundBalerExtension.hasBaleToUnload = true;

		roundBaler.speedLimit = roundBaler.speedLimitBackup;
	end;
end;

function RoundBalerExtension:setCruiseControlWasOn(cruiseControlWasOn, noEventSend)
	self.spec_roundBalerExtension.cruiseControlWasOn = cruiseControlWasOn;

	RoundBalerExtensionCruiseControlEvent.sendEvent(self, cruiseControlWasOn, noEventSend);
end;

function RoundBalerExtension:onReadStream(streamId, connection)
	local specRoundBalerExtension = self.spec_roundBalerExtension;

	specRoundBalerExtension.cruiseControlWasOn = streamReadBool(streamId);
end;

function RoundBalerExtension:onWriteStream(streamId, connection)
	local specRoundBalerExtension = self.spec_roundBalerExtension;

	streamWriteBool(streamId, specRoundBalerExtension.cruiseControlWasOn);
end;

--## MP Event
RoundBalerExtensionCruiseControlEvent = {};
RoundBalerExtensionCruiseControlEvent_mt = Class(RoundBalerExtensionCruiseControlEvent, Event);

InitEventClass(RoundBalerExtensionCruiseControlEvent, "RoundBalerExtensionCruiseControlEvent");

function RoundBalerExtensionCruiseControlEvent.emptyNew()
	local self = Event.new(RoundBalerExtensionCruiseControlEvent_mt);
	
    self.className = "RoundBalerExtensionCruiseControlEvent";
	
	return self;
end;

function RoundBalerExtensionCruiseControlEvent.new(rootVehicle, cruiseControlWasOn)
	local self = RoundBalerExtensionCruiseControlEvent.emptyNew();
	
	self.rootVehicle = rootVehicle;
    self.cruiseControlWasOn = cruiseControlWasOn;
	
	return self;
end;

function RoundBalerExtensionCruiseControlEvent:readStream(streamId, connection)
	self.rootVehicle = NetworkUtil.readNodeObject(streamId);
	self.cruiseControlWasOn = streamReadBool(streamId);
	
    self:run(connection);
end;

function RoundBalerExtensionCruiseControlEvent:writeStream(streamId, connection)
	NetworkUtil.writeNodeObject(streamId, self.rootVehicle);

    streamWriteBool(streamId, self.cruiseControlWasOn);
end;

function RoundBalerExtensionCruiseControlEvent:run(connection)
    if not connection:getIsServer() then
        g_server:broadcastEvent(self, nil, connection, self.rootVehicle);
	end;
	
	if self.rootVehicle ~= nil then
		self.rootVehicle:setCruiseControlWasOn(self.cruiseControlWasOn, true);
	end;
end;

function RoundBalerExtensionCruiseControlEvent.sendEvent(rootVehicle, cruiseControlWasOn, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(RoundBalerExtensionCruiseControlEvent.new(rootVehicle, cruiseControlWasOn), nil, nil, rootVehicle);
		else
			g_client:getServerConnection():sendEvent(RoundBalerExtensionCruiseControlEvent.new(rootVehicle, cruiseControlWasOn));
		end;
	end;
end;